/*
   iomacros.h

   Contributors:
     Created by Marek Michalkiewicz <marekm@linux.org.pl>
     Modified by Don Carveth Jan. 2002.  Added compatibility I/O macros
     for assignment mode of addressing I/O registers.  don@botgoodies.com

   THIS SOFTWARE IS NOT COPYRIGHTED

   This source code is offered for use in the public domain.  You may
   use, modify or distribute it freely.

   This code is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY.  ALL WARRANTIES, EXPRESS OR IMPLIED ARE HEREBY
   DISCLAIMED.  This includes but is not limited to warranties of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef _IOMACROS_H_
#define _IOMACROS_H_

#include <inttypes.h>

#ifndef BV
  #define BV(x) (1 << (x))
#endif

/* Select macro definitions based on I/O Register access method.  Assignment mode 
    allows  register access by direct assignment (PORTA = x or x = PORTA) 
    or by IO macros.    Macro mode must use IO macros to access I/O registers.  */
#ifdef _IO_REG_MACRO_MODE_ 

/*  Macros for MACRO MODE of I/O register access */

/* Memory mapped I/O - use for non-constant I/O space addresses.  */

#define __mmio(port) (*((volatile uint8_t *)((uint8_t)(port) + 0x20)))

/* Macros for access to 8-bit I/O registers with constant address.  */

#define __inb(port) ({				\
	uint8_t __t;				\
	__asm__ __volatile__ (			\
		"in %0,%1"			\
		: "=r" (__t)			\
		: "I" ((uint8_t)(port))		\
	);					\
	__t;					\
 })

/* XXX - order of parameters is confusing, but it is probably too late
   to change now, should be (port, val) - destination on the left side,
   as in memset and C assignment operators.  */

#define __outb(val, port)			\
	__asm__ __volatile__ (			\
		"out %1,%0"			\
		: /* no outputs */		\
		: "r" ((uint8_t)(val)),		\
		  "I" ((uint8_t)(port))		\
	)

#define __outb_zero(port)			\
	__asm__ __volatile__ (			\
		"out %0,__zero_reg__"		\
		: /* no outputs */		\
		: "I" ((uint8_t)(port))		\
	)

/* Macros for access to 8-bit I/O registers (auto detect const address).  */

#define inp(port) (__builtin_constant_p((port)) ? __inb((port)) : __mmio((port)))

#define outp(val,port) do {				\
	if (__builtin_constant_p((port))) {		\
		if (__builtin_constant_p((val)) && ((val) == 0)) \
			__outb_zero((port));		\
		else					\
			__outb((val), (port));		\
	} else						\
		__mmio((port)) = (val);			\
 } while (0)

/* Macros for access to 16-bit I/O registers (ADC, ICR1, OCR1, TCNT1) -
   read low byte first, then high byte (from the TEMP register),
   write the high byte first (to the TEMP register), then the low byte.
   If used from interrupts as well as from the main program, disable
   interrupts first, or use the *_atomic versions (avoid interrupts
   in the middle of operation - there is only one TEMP register).  */

#define __inw(port) ({					\
	uint16_t __t;					\
	__asm__ __volatile__ (				\
		"in %A0,%1" "\n\t"			\
		"in %B0,(%1)+1"				\
		: "=r" (__t)				\
		: "I" ((uint8_t)(port))			\
	);						\
	__t;						\
 })

#define __inw_atomic(port) ({				\
	uint16_t __t;					\
	__asm__ __volatile__ (				\
		"in __tmp_reg__,__SREG__" "\n\t"	\
		"cli" "\n\t"				\
		"in %A0,%1" "\n\t"			\
		"out __SREG__,__tmp_reg__" "\n\t"	\
		"in %B0,(%1)+1"				\
		: "=r" (__t)				\
		: "I" ((uint8_t)(port))			\
		: "r0"					\
	);						\
	__t;						\
 })

#define __outw(val, port)				\
	__asm__ __volatile__ (				\
		"out (%1)+1,%B0" "\n\t"			\
		"out %1,%A0"				\
		: /* no outputs */			\
		: "r" ((uint16_t)(val)),		\
		  "I" ((uint8_t)(port))			\
	)

#define __outw_atomic(val, port)			\
	__asm__ __volatile__ (				\
		"in __tmp_reg__,__SREG__" "\n\t"	\
		"cli" "\n\t"				\
		"out (%1)+1,%B0" "\n\t"			\
		"out __SREG__,__tmp_reg__" "\n\t"	\
		"out %1,%A0"				\
		: /* no outputs */			\
		: "r" ((uint16_t)(val)),		\
		  "I" ((uint8_t)(port))			\
		: "r0"					\
	)

/* __cbi / __sbi require constant port < 0x20 and constant bit */

#define __cbi(port, bit)				\
	__asm__ __volatile__ (				\
		"cbi %0,%1"				\
		: /* no outputs */			\
		: "I" ((uint8_t)(port)),		\
		  "I" ((uint8_t)(bit))			\
	)

#define __sbi(port, bit)				\
	__asm__ __volatile__ (				\
		"sbi %0,%1"				\
		: /* no outputs */			\
		: "I" ((uint8_t)(port)),		\
		  "I" ((uint8_t)(bit))			\
	)

/* __port_and / __port_or work with any constant port */

#define __port_and(port, val)				\
	__asm__ __volatile__ (				\
		"in __tmp_reg__,%0" "\n\t"		\
		"and __tmp_reg__,%1" "\n\t"		\
		"out %0,__tmp_reg__"			\
		: /* no outputs */			\
		: "I" ((uint8_t)(port)),		\
		  "r" ((uint8_t)(val))			\
		: "r0"					\
	)

#define __port_or(port, val)				\
	__asm__ __volatile__ (				\
		"in __tmp_reg__,%0" "\n\t"		\
		"or __tmp_reg__,%1" "\n\t"		\
		"out %0,__tmp_reg__"			\
		: /* no outputs */			\
		: "I" ((uint8_t)(port)),		\
		  "r" ((uint8_t)(val))			\
		: "r0"					\
	)

/* __cbi_const_port / __sbi_const_port work with const or non-const bit */

#define __cbi_const_port(port, bit) do {		\
	if (((port) < 0x20) && __builtin_constant_p((bit))) \
		__cbi((port), (bit));			\
	else						\
		__port_and((port), ~BV((bit)));		\
 } while (0)

#define __sbi_const_port(port, bit) do {		\
	if (((port) < 0x20) && __builtin_constant_p((bit))) \
		__sbi((port), (bit));			\
	else						\
		__port_or((port), BV((bit)));		\
 } while (0)

/* General cbi / sbi macros - work with any (const or non-const) data.  */

#define cbi(port, bit) do {				\
	if (__builtin_constant_p((port)))		\
		__cbi_const_port((port), (bit));	\
	else						\
		__mmio((port)) &= ~BV((bit));		\
 } while (0)

#define sbi(port, bit) do {				\
	if (__builtin_constant_p((port)))		\
		__sbi_const_port((port), (bit));	\
	else						\
		__mmio((port)) |= BV((bit));		\
 } while (0)

#if 1

/* these might make better optimized code? */
#define bit_is_set(port, bit) (__inb(port) & (1<<(bit)))
#define bit_is_clear(port, bit) (!bit_is_set(port, bit))

#else

#define bit_is_clear(port, bit) ({			\
	uint8_t __t;					\
	__asm__ __volatile__ (				\
		"clr %0" "\n\t"				\
		"sbis %1,%2" "\n\t"			\
		"inc %0"				\
		: "=r" (__t)				\
		: "I" ((uint8_t)(port)),		\
		  "I" ((uint8_t)(bit))			\
	);						\
	__t;						\
 })

#define bit_is_set(port, bit) ({			\
	uint8_t __t;					\
	__asm__ __volatile__ (				\
		"clr %0" "\n\t"				\
		"sbic %1,%2" "\n\t"			\
		"inc %0"				\
		: "=r" (__t)				\
		: "I" ((uint8_t)(port)),		\
		  "I" ((uint8_t)(bit))			\
	);						\
	__t;						\
 })

#endif

#define loop_until_bit_is_set(port, bit)		\
	__asm__ __volatile__ (				\
	"L_%=: " "sbis %0,%1" "\n\t"			\
		"rjmp L_%="				\
		: /* no outputs */			\
		: "I" ((uint8_t)(port)),		\
		  "I" ((uint8_t)(bit))			\
	)

#define loop_until_bit_is_clear(port, bit)		\
	__asm__ __volatile__ (				\
	"L_%=: " "sbic %0,%1" "\n\t"			\
		"rjmp L_%="				\
		: /* no outputs */			\
		: "I" ((uint8_t)(port)),		\
		  "I" ((uint8_t)(bit))			\
	)

#define parity_even_bit(val) ({				\
	uint8_t __t;					\
	__asm__ (					\
		"mov __tmp_reg__,%0" "\n\t"		\
		"swap %0" "\n\t"			\
		"eor %0,__tmp_reg__" "\n\t"		\
		"mov __tmp_reg__,%0" "\n\t"		\
		"lsr %0" "\n\t"				\
		"lsr %0" "\n\t"				\
		"eor %0,__tmp_reg__" "\n\t"		\
		"mov __tmp_reg__,%0" "\n\t"		\
		"lsr %0" "\n\t"				\
		"eor %0,__tmp_reg__"			\
		: "=r" (__t)				\
		: "0" ((uint8_t)(val))			\
		: "r0"					\
	);						\
	(__t & 1);					\
 })

#else

/* Macros for ASSIGNMENT MODE of I/O Register Access */

/* In this mode the registers are treated as any other RAM address, the port names
are defined as:  #define PORTA	(*(volatile unsigned char *)0x3B).  Using
this  method a reference to PORTA in your code refers to the value
stored at memory location 0x38.  This allows direct I/O register assignment 
such as  PORTA = x or x = PORTA.

The macros below are included primarily for compatability mode with Mareks
original macros, with the exception of the bit operation macros for which
direct assignment does not work.  Additional versions of the sbi and cbi macros,
which allow the use of port and bit variables, but are slightly slower, are included 
as sbiv and cbiv.

Compatibilty - all macros are compatible with the Mareks original 
version above and compile to essentially the same code.  Versions of some macros
with leading underscores  are included for compatibility.

Added by Don Carveth, Jan. 2002
*/

/* inp,  inb and __inb are defined and are identical.
    outp, outb,  __outb are defined and are identical. 
    The compiler is smart enough to know if a port is being referenced
    and then uses in or out instructions */
#define inp(port)   (port)
#define inb(port)   (port)
#define outp(val, port)   port = val
#define outb(val, port)   port = val
#define __outb_zero(port)  port = 0
#define __inb inb
#define __outb   outb

/*  Word (16 bit) operators. */
/*  The compiler produces the proper in / out instructions and reads or writes
     the bytes in the correct order  */
#define inw(port) (port)
#define outw(val,port)   port = val
#define __inw  inw
#define __outw  outw

/*  Bit operations  */
/*  bit must be a constant */

#define cbi(port, bit)				\
	__asm__ __volatile__ (				\
		"cbi %0,%1"				\
		: /* no outputs */			\
		: "I" ((uint8_t)&(port) - 0x20),		\
		  "I" ((uint8_t)(bit))			\
	)

#define sbi(port, bit)				\
	__asm__ __volatile__ (				\
		"sbi %0,%1"				\
		: /* no outputs */			\
		: "I" ((uint8_t)&(port) - 0x20),		\
		  "I" ((uint8_t)(bit))			\
	)
   
#define __cbi  cbi
#define __sbi  sbi   
    
#define bit_is_set(port,bit)     (port & (BV(bit)))
#define bit_is_clear(port,bit)    (!bit_is_set(port, bit))
#define loop_until_bit_is_set(port,bit)    while (bit_is_clear(port, bit))
#define loop_until_bit_is_clear(port,bit)    while (bit_is_set(port, bit))

#define port_or(port, val)   port | val
#define port_and(port, val)   port & val
#define __port_or  port_or
#define __port_and  port_and

/*  Bit operations with variables
     The cbiv, sbiv are slower than the cbi, sbi above but allow the use of 
     variables for port and bit.  ( 3 cycles compared to 2,  3 words compared
     to 1  */
#define cbiv(port,bit)    port &= ~BV(bit) 
#define sbiv(port,bit)    port |= BV(bit)

/* I did not include the parity_even_bit macro as I am not sure what it does */

#endif /* _IO_REG_ASSIGN_MODE_   */
#endif /* _IOMACROS_H_ */
